<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'GT_BOGO_ENGINE_VERSION', '3.5.17' );
define( 'GT_BOGO_ENGINE_FILE', __FILE__ );
define( 'GT_BOGO_ENGINE_PATH', plugin_dir_path( __FILE__ ) );
define( 'GT_BOGO_ENGINE_URL', plugin_dir_url( __FILE__ ) );
define( 'GT_BOGO_ENGINE_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Plugin Name: GT BOGO ENGINE for WooCommerce
 * Plugin URI: https://graphictshirts.shop/bogo
 * Description: Smart Buy X Get Y (BOGO) promotions for WooCommerce with a visual rules builder. Includes a free global BOGO rule plus advanced rule logic.
 * Version: 3.5.17
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Author: GRAPHIC T-SHIRTS
 * Author URI: https://graphictshirts.shop
 * Text Domain: gt-bogo-engine
 * Domain Path: /languages
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// -----------------------------------------------------------------------------
// Core Constants (bootstrap locked)
// -----------------------------------------------------------------------------
if ( ! defined( 'GT_BOGO_ENGINE_MENU_SLUG' ) ) {
    define( 'GT_BOGO_ENGINE_MENU_SLUG', 'gt-bogo-engine' );
}
if ( ! defined( 'GT_BOGO_ENGINE_PATH' ) ) {
    define( 'GT_BOGO_ENGINE_PATH', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'GT_BOGO_ENGINE_URL' ) ) {
    define( 'GT_BOGO_ENGINE_URL', plugin_dir_url( __FILE__ ) );
}



if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * -------------------------------------------------------------------------
 *  CONSTANTS
 * -------------------------------------------------------------------------
 */
if ( ! defined( 'GT_BOGO_ENGINE_VERSION' ) ) {
    define( 'GT_BOGO_ENGINE_VERSION', '3.5.17' );
}

if ( ! defined( 'GT_BOGO_ENGINE_UPDATE_API' ) ) {
    define( 'GT_BOGO_ENGINE_UPDATE_API', 'https://graphictshirts.shop/downloads/?gt-bogo-engine-update=1' );
}

if ( ! defined( 'GT_BOGO_ENGINE_FILE' ) ) {
    define( 'GT_BOGO_ENGINE_FILE', __FILE__ );
}

if ( ! defined( 'GT_BOGO_ENGINE_DIR' ) ) {
    define( 'GT_BOGO_ENGINE_DIR', plugin_dir_path( __FILE__ ) );
}

if ( ! defined( 'GT_BOGO_ENGINE_URL' ) ) {
    define( 'GT_BOGO_ENGINE_URL', plugin_dir_url( __FILE__ ) );
}

/**
 * -------------------------------------------------------------------------
 *  LOAD TEXTDOMAIN (TRANSLATIONS)
 * -------------------------------------------------------------------------
 */
function gt_bogo_engine_load_textdomain() {
    load_plugin_textdomain(
        'gt-bogo-engine',
        false,
        dirname( plugin_basename( __FILE__ ) ) . '/languages'
    );
}
add_action( 'init', 'gt_bogo_engine_load_textdomain' );

/**
 * -------------------------------------------------------------------------
 *  REQUIRE CORE ENGINE
 * -------------------------------------------------------------------------
 *
 * The core engine then pulls in:
 *  - includes/class-gt-bogo-engine-admin.php
 *  - includes/class-gt-bogo-engine-cart.php
 *  - and any other supporting classes.
 */
require_once GT_BOGO_ENGINE_DIR . 'includes/class-gt-bogo-engine.php';
require_once GT_BOGO_ENGINE_PATH . 'includes/class-gt-bogo-engine-license.php';

/**
 * -------------------------------------------------------------------------
 *  BOOTSTRAP PLUGIN (after WooCommerce is loaded)
 * -------------------------------------------------------------------------
 */
function gt_bogo_engine_init() {

    // WooCommerce dependency check.
    if ( ! class_exists( 'WooCommerce' ) ) {

        add_action( 'admin_notices', function () {
            if ( ! current_user_can( 'activate_plugins' ) ) {
                return;
            }

            echo '<div class="notice notice-error"><p>';
            echo esc_html__(
                'GT BOGO ENGINE requires WooCommerce to be installed and active.',
                'gt-bogo-engine'
            );
            echo '</p></div>';
        } );

        return;
    }

    // Initialize main engine singleton.
    if ( class_exists( 'GT_BOGO_Engine' ) ) {
        GT_BOGO_Engine::instance();
    }
}
add_action( 'plugins_loaded', 'gt_bogo_engine_init', 20 );

/**
 * -------------------------------------------------------------------------
 *  ACTIVATION HOOK
 * -------------------------------------------------------------------------
 *
 * Sets up sane defaults so store owners see something working immediately.
 */


/**
 * Flush all relevant caches when GT BOGO ENGINE is installed, updated or deactivated.
 *
 * This clears WordPress object cache, GT BOGO ENGINE transients, any
 * plugin-specific cache directory under uploads, and PHP OPcache when available.
 */
function gt_bogo_engine_flush_all_caches() {
	// Flush WordPress object cache if available.
	if ( function_exists( 'wp_cache_flush' ) ) {
		wp_cache_flush();
	}

	// Clear GT BOGO ENGINE related transients.
	global $wpdb;
	if ( isset( $wpdb ) ) {
		$like = $wpdb->esc_like( 'gt_bogo_engine_' ) . '%';
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
				'_transient_' . $like,
				'_transient_timeout_' . $like
			)
		);
	}

	// Clear any plugin-specific cache directory if it exists.
	if ( defined( 'WP_CONTENT_DIR' ) ) {
		$cache_dir = trailingslashit( WP_CONTENT_DIR ) . 'uploads/gt-bogo-engine-cache/';
		if ( is_dir( $cache_dir ) ) {
			// Use WP_Filesystem when available.
			if ( ! function_exists( 'WP_Filesystem' ) ) {
				require_once ABSPATH . 'wp-admin/includes/file.php';
			}
			WP_Filesystem();
			global $wp_filesystem;
			if ( $wp_filesystem ) {
				$wp_filesystem->delete( $cache_dir, true );
			}
		}
	}

	// Reset OPcache if available.
	if ( function_exists( 'opcache_reset' ) ) {
		@opcache_reset();
	}
}


function gt_bogo_engine_activate() {
	gt_bogo_engine_flush_all_caches();

    // Base plugin settings.
    if ( ! get_option( 'gt_bogo_engine_settings' ) ) {
        update_option(
            'gt_bogo_engine_settings',
            array(
                'enable_engine'  => 1,
                'debug_mode'     => 0,
                'pro_nag_hidden' => 0,
            )
        );
    }

    // Free global rule: Buy 1 Get 1 at 50% off (global scope).
    if ( ! get_option( 'gt_bogo_engine_free_rule' ) ) {
        update_option(
            'gt_bogo_engine_free_rule',
            array(
                'enabled'        => 1,
                'title'          => __( 'Buy 1 Get 1 at 50% Off (Global)', 'gt-bogo-engine' ),
                'buy_qty'        => 1,
                'get_qty'        => 1,
                'discount_type'  => 'percent',
                'discount_value' => 50,
                'scope'          => 'global', // global | categories | products
                'categories'     => array(),
                'products'       => array(),
            )
        );
    }

    // PRO rules array (empty by default).
    if ( ! get_option( 'gt_bogo_engine_rules' ) ) {
        update_option( 'gt_bogo_engine_rules', array() );
    }

    // You can also pre-seed a sample PRO rule if desired:
    // $rules = get_option( 'gt_bogo_engine_rules', array() );
    // $rules[] = array(
    //     'enabled'        => 0,
    //     'title'          => __( 'Sample PRO Rule – Buy 2 Get 1 Free', 'gt-bogo-engine' ),
    //     'buy_qty'        => 2,
    //     'get_qty'        => 1,
    //     'discount_type'  => 'free_item',
    //     'discount_value' => 100,
    //     'scope'          => 'global',
    //     'categories'     => array(),
    //     'products'       => array(),
    // );
    // update_option( 'gt_bogo_engine_rules', $rules );
}
/**
 * (Some installs/theme setups hide the column or users miss it; this makes it explicit.)
 */
add_filter( 'plugin_action_links_' . GT_BOGO_ENGINE_BASENAME, function( $links ) {
    if ( ! is_admin() ) {
        return $links;
    }
    if ( ! current_user_can( 'update_plugins' ) ) {
        return $links;
    }

    $plugin = GT_BOGO_ENGINE_BASENAME;

    // Detect current per-plugin auto-update setting.
    $enabled = false;
    $list = (array) get_site_option( 'auto_update_plugins', array() );
    if ( in_array( $plugin, $list, true ) ) {
        $enabled = true;
    }

    if ( $enabled ) {
        $url = wp_nonce_url(
            self_admin_url( 'plugins.php?action=disable-auto-update&plugin=' . rawurlencode( $plugin ) ),
            'disable-auto-update-plugin_' . $plugin
        );
    } else {
        $url = wp_nonce_url(
            self_admin_url( 'plugins.php?action=enable-auto-update&plugin=' . rawurlencode( $plugin ) ),
            'enable-auto-update-plugin_' . $plugin
        );
    }

    return $links;
}, 50 );


register_activation_hook( __FILE__, 'gt_bogo_engine_activate' );

/**
 * -------------------------------------------------------------------------
 *  DEACTIVATION HOOK
 * -------------------------------------------------------------------------
 *
 * We do NOT delete data here. Uninstall is handled via uninstall.php
 * so site owners can safely test and reactivate.
 */
function gt_bogo_engine_deactivate() {
	gt_bogo_engine_flush_all_caches();
    // Intentionally left blank – configuration lives until uninstall.php is run.
}
register_deactivation_hook( __FILE__, 'gt_bogo_engine_deactivate' );






/**
 * When the plugin is updated via the standard WordPress updater,
 * force all caches to be cleared so the new UI and code are loaded immediately.
 *
 * @param WP_Upgrader $upgrader
 * @param array       $options
 */
function gt_bogo_engine_on_upgrade( $upgrader, $options ) {
	if ( empty( $options['type'] ) || 'plugin' !== $options['type'] ) {
		return;
	}

	if ( empty( $options['plugins'] ) || ! is_array( $options['plugins'] ) ) {
		return;
	}

	$plugin_basename = plugin_basename( __FILE__ );
	if ( in_array( $plugin_basename, $options['plugins'], true ) ) {
		gt_bogo_engine_flush_all_caches();
	}
}
add_action( 'upgrader_process_complete', 'gt_bogo_engine_on_upgrade', 10, 2 );


/**
 * Register GT BOGO ENGINE with WordPress' update system using a private endpoint
 * hosted on graphictshirts.shop.
 *
 * This allows customer sites to receive one‑click updates whenever a new
 * ZIP is uploaded to the /downloads directory on the update server.
 */

/**
 * Inject update information into the plugin update transient.
 *
 * @param stdClass $transient
 * @return stdClass
 */
function gt_bogo_engine_check_for_updates( $transient ) {
	if ( empty( $transient->checked ) ) {
		return $transient;
	}

	if ( ! defined( 'GT_BOGO_ENGINE_UPDATE_API' ) ) {
		return $transient;
	}

	$plugin_basename = plugin_basename( __FILE__ );

	$current_version = GT_BOGO_ENGINE_VERSION;
	$request_body    = array(
		'action'  => 'gt_bogo_engine_check_update',
		'slug'    => 'gt-bogo-engine',
		'license' => '', // Reserved for future license handling.
		'url'     => home_url(),
		'version' => $current_version,
	);

	$response = wp_remote_post(
		GT_BOGO_ENGINE_UPDATE_API,
		array(
			'timeout' => 15,
			'body'    => $request_body,
		)
	);

	if ( is_wp_error( $response ) ) {
		return $transient;
	}

	$code = wp_remote_retrieve_response_code( $response );
	if ( 200 !== $code ) {
		return $transient;
	}

	$body = wp_remote_retrieve_body( $response );
	$data = json_decode( $body );

	if ( empty( $data ) || empty( $data->new_version ) ) {
		return $transient;
	}

	if ( version_compare( $current_version, $data->new_version, '>=' ) ) {
		return $transient;
	}

	$update              = new stdClass();
	$update->slug        = 'gt-bogo-engine';
	$update->plugin      = $plugin_basename;
	$update->new_version = $data->new_version;
	$update->package     = ! empty( $data->package ) ? $data->package : '';
	$update->url         = ! empty( $data->url ) ? $data->url : 'https://graphictshirts.shop';
	$update->tested      = ! empty( $data->tested ) ? $data->tested : '';
	$update->requires    = ! empty( $data->requires ) ? $data->requires : '';

	$transient->response[ $plugin_basename ] = $update;

	return $transient;
}
add_filter( 'pre_set_site_transient_update_plugins', 'gt_bogo_engine_check_for_updates' );

/**
 * Provide plugin information for the "View details" modal in the Plugins screen.
 *
 * @param false|object|array $result
 * @param string             $action
 * @param object             $args
 *
 * @return false|object|array
 */
function gt_bogo_engine_plugins_api( $result, $action, $args ) {
	if ( 'plugin_information' !== $action || empty( $args->slug ) || 'gt-bogo-engine' !== $args->slug ) {
		return $result;
	}

	if ( ! defined( 'GT_BOGO_ENGINE_UPDATE_API' ) ) {
		return $result;
	}

	$request_body = array(
		'action' => 'gt_bogo_engine_plugin_information',
		'slug'   => 'gt-bogo-engine',
	);

	$response = wp_remote_post(
		GT_BOGO_ENGINE_UPDATE_API,
		array(
			'timeout' => 15,
			'body'    => $request_body,
		)
	);

	if ( is_wp_error( $response ) ) {
		return $result;
	}

	$code = wp_remote_retrieve_response_code( $response );
	if ( 200 !== $code ) {
		return $result;
	}

	$body = wp_remote_retrieve_body( $response );
	$data = json_decode( $body );

	if ( empty( $data ) ) {
		return $result;
	}

	return $data;
}
add_filter( 'plugins_api', 'gt_bogo_engine_plugins_api', 10, 3 );



// -----------------------------------------------------------------------------
// Lifecycle hardening: cache clearing, update integrity, and safe uninstall cleanup
// -----------------------------------------------------------------------------

/**
 * Purge GT BOGO ENGINE caches (safe for shared hosting; no shell commands).
 * Note: This clears transients/object cache/OPcache and common cache plugins if present.
 */
function gt_bogo_engine_purge_all_caches() {
    global $wpdb;

    // Plugin transients only (do NOT delete license/settings options here)
    $like = $wpdb->esc_like( 'gt_bogo_' ) . '%';
    $wpdb->query( $wpdb->prepare(
        "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
        '_transient_' . $like,
        '_transient_timeout_' . $like
    ) );
    $wpdb->query( $wpdb->prepare(
        "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
        '_site_transient_' . $like,
        '_site_transient_timeout_' . $like
    ) );

    // Flush WP object cache
    if ( function_exists( 'wp_cache_flush' ) ) {
        @wp_cache_flush();
    }

    // Flush rewrite rules (safe)
    if ( function_exists( 'flush_rewrite_rules' ) ) {
        @flush_rewrite_rules();
    }

    // Common cache plugins (only if installed)
    if ( function_exists( 'rocket_clean_domain' ) ) { @rocket_clean_domain(); }
    if ( function_exists( 'w3tc_flush_all' ) ) { @w3tc_flush_all(); }
    if ( function_exists( 'wp_cache_clear_cache' ) ) { @wp_cache_clear_cache(); } // WP Super Cache
    if ( class_exists( 'LiteSpeed\Purge' ) ) { try { \LiteSpeed\Purge::purge_all(); } catch ( \Throwable $e ) {} }
    if ( function_exists( 'sg_cachepress_purge_cache' ) ) { @sg_cachepress_purge_cache(); }
    if ( function_exists( 'cache_enabler_clear_total_cache' ) ) { @cache_enabler_clear_total_cache(); }

    // PHP OPcache (if enabled/allowed)
    if ( function_exists( 'opcache_reset' ) ) {
        @opcache_reset();
    }
}

function gt_bogo_engine_on_activate() {
    update_option( 'gt_bogo_engine_installed_version', GT_BOGO_ENGINE_VERSION, false );
    gt_bogo_engine_purge_all_caches();
}

function gt_bogo_engine_on_deactivate() {
    gt_bogo_engine_purge_all_caches();
}

register_activation_hook( __FILE__, 'gt_bogo_engine_on_activate' );
register_deactivation_hook( __FILE__, 'gt_bogo_engine_on_deactivate' );

// After plugin update, purge caches immediately
add_action( 'upgrader_process_complete', function( $upgrader, $options ) {
    if ( empty( $options['type'] ) || 'plugin' !== $options['type'] ) return;
    if ( empty( $options['plugins'] ) || ! is_array( $options['plugins'] ) ) return;
    if ( in_array( plugin_basename( __FILE__ ), $options['plugins'], true ) ) {
        update_option( 'gt_bogo_engine_installed_version', GT_BOGO_ENGINE_VERSION, false );
        gt_bogo_engine_purge_all_caches();
    }
}, 10, 2 );

// Safety net: purge once if version changed but host did not trigger updater hook
add_action( 'admin_init', function() {
    $installed = get_option( 'gt_bogo_engine_installed_version', '' );
    if ( $installed !== GT_BOGO_ENGINE_VERSION ) {
        update_option( 'gt_bogo_engine_installed_version', GT_BOGO_ENGINE_VERSION, false );
        gt_bogo_engine_purge_all_caches();
    }
} );

// Keep GT BOGO pages clean: hide third-party admin notices ONLY on our plugin screens (prevents UI bleed)
add_action( 'current_screen', function( $screen ) {
    if ( ! is_object( $screen ) || empty( $screen->id ) ) return;
    $id = (string) $screen->id;

    $is_gt_page = ( false !== strpos( $id, 'gt-bogo-engine' ) );
    if ( defined( 'GT_BOGO_ENGINE_MENU_SLUG' ) && GT_BOGO_ENGINE_MENU_SLUG ) {
        $is_gt_page = $is_gt_page || ( false !== strpos( $id, GT_BOGO_ENGINE_MENU_SLUG ) );
    }

    if ( $is_gt_page ) {
        add_action( 'in_admin_header', function() {
            remove_all_actions( 'admin_notices' );
            remove_all_actions( 'all_admin_notices' );
        }, 0 );
    }
} );
