<?php
/**
 * Admin menu + page router for GT BOGO Engine.
 *
 * @package GT_BOGO_Engine
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class GT_BOGO_Admin_Menu {

	/**
	 * Initialize hooks.
	 */
	public static function init() {
		add_action( 'admin_menu', array( __CLASS__, 'register_menu' ) );
		add_filter( 'plugin_action_links_' . plugin_basename( GT_BOGO_ENGINE_FILE ), array( __CLASS__, 'plugin_action_links' ) );
	}

	/**
	 * Register top level menu + submenus.
	 */
	public static function register_menu() {

		$capability = 'manage_options';

		// Top level.
		add_menu_page(
			__( 'GT BOGO Engine', 'gt-bogo-engine' ),
			__( 'GT BOGO Engine', 'gt-bogo-engine' ),
			$capability,
			'gt-bogo-engine',
			array( __CLASS__, 'render_dashboard' ),
			'dashicons-tag',
			56
		);

		// Dashboard (main).
		add_submenu_page(
			'gt-bogo-engine',
			__( 'Dashboard', 'gt-bogo-engine' ),
			__( 'Dashboard', 'gt-bogo-engine' ),
			$capability,
			'gt-bogo-engine',
			array( __CLASS__, 'render_dashboard' )
		);

		// Free rule.
		add_submenu_page(
			'gt-bogo-engine',
			__( 'Free Global Rule', 'gt-bogo-engine' ),
			__( 'Free Rule', 'gt-bogo-engine' ),
			$capability,
			'gt-bogo-engine-free',
			array( __CLASS__, 'render_free_rule' )
		);

		// PRO rules.
		add_submenu_page(
			'gt-bogo-engine',
			__( 'PRO Rules', 'gt-bogo-engine' ),
			__( 'PRO Rules', 'gt-bogo-engine' ),
			$capability,
			'gt-bogo-engine-pro',
			array( __CLASS__, 'render_pro_rules' )
		);

		// Settings.
		add_submenu_page(
			'gt-bogo-engine',
			__( 'Settings', 'gt-bogo-engine' ),
			__( 'Settings', 'gt-bogo-engine' ),
			$capability,
			'gt-bogo-engine-settings',
			array( __CLASS__, 'render_settings' )
		);

		// Logs.
		add_submenu_page(
			'gt-bogo-engine',
			__( 'Logs', 'gt-bogo-engine' ),
			__( 'Logs', 'gt-bogo-engine' ),
			$capability,
			'gt-bogo-engine-logs',
			array( __CLASS__, 'render_logs' )
		);

		// Help.
		add_submenu_page(
			'gt-bogo-engine',
			__( 'Help & About', 'gt-bogo-engine' ),
			__( 'Help & About', 'gt-bogo-engine' ),
			$capability,
			'gt-bogo-engine-help',
			array( __CLASS__, 'render_help' )
		);
	}

	/**
	 * "Settings" link on the Plugins screen.
	 *
	 * @param array $links
	 *
	 * @return array
	 */
	public static function plugin_action_links( $links ) {
		$settings_url = admin_url( 'admin.php?page=gt-bogo-engine-settings' );
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			esc_url( $settings_url ),
			esc_html__( 'Settings', 'gt-bogo-engine' )
		);

		array_unshift( $links, $settings_link );

		$docs_link = sprintf(
			'<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
			esc_url( 'https://graphictshirts.shop/bogo' ),
			esc_html__( 'Docs & PRO', 'gt-bogo-engine' )
		);

		$links[] = $docs_link;

		return $links;
	}

	/**
	 * Dashboard page.
	 */
	public static function render_dashboard() {
		$settings   = get_option( 'gt_bogo_engine_settings', array() );
		$free_rule  = get_option( 'gt_bogo_engine_free_rule', array() );
		$pro_rules  = get_option( 'gt_bogo_engine_rules', array() );
		$engine_on  = ! empty( $settings['enable_engine'] );
		$pro_count  = is_array( $pro_rules ) ? count( $pro_rules ) : 0;
		$free_title = isset( $free_rule['title'] ) ? $free_rule['title'] : __( 'Buy 1 Get 1 at 50% Off (Global)', 'gt-bogo-engine' );
		?>
		<div class="wrap gt-bogo-wrap">
			<h1 class="wp-heading-inline">
				<?php esc_html_e( 'GT BOGO Rules Engine for WooCommerce', 'gt-bogo-engine' ); ?>
			</h1>
			<hr class="wp-header-end" />

			<div class="gt-bogo-dashboard-grid">
				<div class="gt-bogo-card">
					<h2><?php esc_html_e( 'Engine Status', 'gt-bogo-engine' ); ?></h2>
					<p>
						<?php if ( $engine_on ) : ?>
							<span class="gt-bogo-status gt-bogo-status-on"><?php esc_html_e( 'Active', 'gt-bogo-engine' ); ?></span>
						<?php else : ?>
							<span class="gt-bogo-status gt-bogo-status-off"><?php esc_html_e( 'Disabled', 'gt-bogo-engine' ); ?></span>
						<?php endif; ?>
					</p>
					<p>
						<?php esc_html_e( 'The engine runs on the WooCommerce cart and applies the configured free and PRO rules automatically.', 'gt-bogo-engine' ); ?>
					</p>
					<p>
						<a class="button button-primary" href="<?php echo esc_url( admin_url( 'admin.php?page=gt-bogo-engine-settings' ) ); ?>">
							<?php esc_html_e( 'Open Settings', 'gt-bogo-engine' ); ?>
						</a>
					</p>
				</div>

				<div class="gt-bogo-card">
					<h2><?php esc_html_e( 'Free Global Rule', 'gt-bogo-engine' ); ?></h2>
					<p><strong><?php echo esc_html( $free_title ); ?></strong></p>
					<p>
						<?php esc_html_e( 'This is the always-free global BOGO rule. Use it to quickly run “Buy 1 Get 1 50% Off” on your entire catalog.', 'gt-bogo-engine' ); ?>
					</p>
					<p>
						<a class="button" href="<?php echo esc_url( admin_url( 'admin.php?page=gt-bogo-engine-free' ) ); ?>">
							<?php esc_html_e( 'Configure Free Rule', 'gt-bogo-engine' ); ?>
						</a>
					</p>
				</div>

				<div class="gt-bogo-card">
					<h2><?php esc_html_e( 'PRO Rules Overview', 'gt-bogo-engine' ); ?></h2>
					<p>
						<?php
						printf(
							/* translators: %d: number of rules */
							esc_html__( '%d PRO rule(s) configured.', 'gt-bogo-engine' ),
							intval( $pro_count )
						);
						?>
					</p>
					<p>
						<?php esc_html_e( 'PRO rules let you create advanced BOGO campaigns: free items, cheapest-item discounts, category-targeted promos, and scheduled offers.', 'gt-bogo-engine' ); ?>
					</p>
					<p>
						<a class="button button-secondary" href="<?php echo esc_url( admin_url( 'admin.php?page=gt-bogo-engine-pro' ) ); ?>">
							<?php esc_html_e( 'Manage PRO Rules', 'gt-bogo-engine' ); ?>
						</a>
						<a class="button button-link" href="https://graphictshirts.shop/bogo" target="_blank" rel="noopener noreferrer">
							<?php esc_html_e( 'Learn about PRO licensing →', 'gt-bogo-engine' ); ?>
						</a>
					</p>
				</div>

				<div class="gt-bogo-card">
					<h2><?php esc_html_e( 'About GRAPHIC T-SHIRTS', 'gt-bogo-engine' ); ?></h2>
					<p>
						<?php esc_html_e( 'GT BOGO Engine is crafted by GRAPHIC T-SHIRTS – an independent premium streetwear brand running WooCommerce at scale.', 'gt-bogo-engine' ); ?>
					</p>
					<p>
						<a class="button button-secondary" href="https://graphictshirts.shop" target="_blank" rel="noopener noreferrer">
							<?php esc_html_e( 'Visit GRAPHIC T-SHIRTS', 'gt-bogo-engine' ); ?>
						</a>
					</p>
				</div>
			</div>

			<style>
				.gt-bogo-dashboard-grid {
					display: grid;
					grid-template-columns: repeat(auto-fit, minmax(260px, 1fr));
					grid-gap: 20px;
					margin-top: 20px;
				}
				.gt-bogo-card {
					background: #ffffff;
					border: 1px solid #e5e7eb;
					border-radius: 8px;
					padding: 16px 18px 18px;
					box-shadow: 0 3px 14px rgba(15, 23, 42, 0.05);
				}
				.gt-bogo-status {
					display: inline-block;
					padding: 3px 10px;
					border-radius: 999px;
					font-size: 11px;
					text-transform: uppercase;
					letter-spacing: .08em;
				}
				.gt-bogo-status-on {
					background: #ecfdf3;
					color: #15803d;
				}
				.gt-bogo-status-off {
					background: #fef2f2;
					color: #b91c1c;
				}
			</style>
		</div>
		<?php
	}

	/**
	 * Free rule page.
	 */
	public static function render_free_rule() {
		if ( class_exists( 'GT_BOGO_Admin_Free_Rule' ) ) {
			GT_BOGO_Admin_Free_Rule::render_page();
			return;
		}

		self::missing_class_notice( 'GT_BOGO_Admin_Free_Rule' );
	}

	/**
	 * PRO rules page.
	 */
	public static function render_pro_rules() {
		if ( class_exists( 'GT_BOGO_Admin_Pro_Rules' ) ) {
			GT_BOGO_Admin_Pro_Rules::render_page();
			return;
		}

		self::missing_class_notice( 'GT_BOGO_Admin_Pro_Rules' );
	}

	/**
	 * Settings page.
	 */
	public static function render_settings() {
		if ( class_exists( 'GT_BOGO_Admin_Settings' ) ) {
			GT_BOGO_Admin_Settings::render_page();
			return;
		}

		self::missing_class_notice( 'GT_BOGO_Admin_Settings' );
	}

	/**
	 * Logs page.
	 */
	public static function render_logs() {
		if ( class_exists( 'GT_BOGO_Admin_Logs' ) ) {
			GT_BOGO_Admin_Logs::render_page();
			return;
		}

		self::missing_class_notice( 'GT_BOGO_Admin_Logs' );
	}

	/**
	 * Help / About page.
	 */
	public static function render_help() {
		?>
		<div class="wrap gt-bogo-wrap">
			<h1><?php esc_html_e( 'GT BOGO Engine – Help & About', 'gt-bogo-engine' ); ?></h1>
			<p>
				<?php esc_html_e( 'This plugin adds Buy X Get Y (BOGO) logic directly into your WooCommerce cart.', 'gt-bogo-engine' ); ?>
			</p>

			<h2><?php esc_html_e( 'Getting Started', 'gt-bogo-engine' ); ?></h2>
			<ol>
				<li><?php esc_html_e( 'Visit the “Free Rule” tab and configure your global BOGO offer.', 'gt-bogo-engine' ); ?></li>
				<li><?php esc_html_e( 'Optionally, create additional PRO rules for specific categories or products.', 'gt-bogo-engine' ); ?></li>
				<li><?php esc_html_e( 'Make sure the engine is enabled in “Settings”.', 'gt-bogo-engine' ); ?></li>
				<li><?php esc_html_e( 'Add items to cart on the front-end and confirm that the discounts apply automatically.', 'gt-bogo-engine' ); ?></li>
			</ol>

			<h2><?php esc_html_e( 'Support & Feedback', 'gt-bogo-engine' ); ?></h2>
			<p>
				<?php esc_html_e( 'If you run into issues or have feature requests, we’d love to hear from you.', 'gt-bogo-engine' ); ?>
			</p>
			<ul>
				<li>
					<strong><?php esc_html_e( 'Plugin Page:', 'gt-bogo-engine' ); ?></strong>
					<a href="https://graphictshirts.shop/bogo" target="_blank" rel="noopener noreferrer">
						https://graphictshirts.shop/bogo
					</a>
				</li>
				<li>
					<strong><?php esc_html_e( 'Brand:', 'gt-bogo-engine' ); ?></strong>
					<a href="https://graphictshirts.shop" target="_blank" rel="noopener noreferrer">
						GRAPHIC T-SHIRTS
					</a>
				</li>
			</ul>
		</div>
		<?php
	}

	/**
	 * Fallback if admin class missing.
	 *
	 * @param string $class_name
	 */
	protected static function missing_class_notice( $class_name ) {
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'GT BOGO Engine', 'gt-bogo-engine' ); ?></h1>
			<div class="notice notice-error">
				<p>
					<?php
					printf(
						/* translators: %s: class name */
						esc_html__( 'Admin class %s is not available. Please reinstall the GT BOGO Engine plugin.', 'gt-bogo-engine' ),
						esc_html( $class_name )
					);
					?>
				</p>
			</div>
		</div>
		<?php
	}
}
