<?php
/**
 * GT BOGO Rules Engine – Free Rule UI & Logic
 *
 * Handles the main free rule:
 * "Buy X Get Y at % Off" with a visual, commercial-grade admin UI.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class GT_BOGO_Engine_Free_Rule {

    private static $instance = null;

    /**
     * Singleton
     */
    public static function instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {

        // Hook into our main admin screen tab system.
        add_action( 'gt_bogo_engine_render_tab_free', array( $this, 'render_page' ) );

        // Handle form submit.
        add_action( 'admin_post_gt_bogo_save_free_rule', array( $this, 'save_rule' ) );
    }

    /**
     * Get stored rule (with sane defaults)
     */
    public function get_rule() {

        $rule = get_option( 'gt_bogo_engine_free_rule', array() );

        $defaults = array(
            'enabled'               => 1,
            'title'                 => 'Buy 1 Get 1 at 50% Off (Global)',
            'buy_qty'               => 1,
            'get_qty'               => 1,
            'discount_type'         => 'percent', // percent|fixed
            'discount_value'        => 50,
            'scope'                 => 'global',  // global|categories|products
            'categories'            => array(),
            'products'              => array(),
            'min_cart_subtotal'     => '',
            'max_discounts_per_cart'=> '',
            'apply_same_product'    => 1,
            'allow_coupon_stack'    => 0,
            'schedule_enabled'      => 0,
            'schedule_start'        => '',
            'schedule_end'          => '',
            'allowed_roles'         => array(),
            'banner_text'           => 'Buy 1 Get 1 at 50% off – applied automatically in cart!',
            'banner_position'       => 'before_cart', // before_cart|after_cart|none
        );

        return wp_parse_args( $rule, $defaults );
    }

    /**
     * Save rule
     */
    public function save_rule() {

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( 'Not allowed.' );
        }

        check_admin_referer( 'gt_bogo_save_free_rule_nonce' );

        $data = array();

        $data['enabled']            = isset( $_POST['enabled'] ) ? 1 : 0;
        $data['title']              = isset( $_POST['title'] ) ? sanitize_text_field( $_POST['title'] ) : '';
        $data['buy_qty']            = isset( $_POST['buy_qty'] ) ? max( 1, intval( $_POST['buy_qty'] ) ) : 1;
        $data['get_qty']            = isset( $_POST['get_qty'] ) ? max( 1, intval( $_POST['get_qty'] ) ) : 1;
        $data['discount_type']      = isset( $_POST['discount_type'] ) && in_array( $_POST['discount_type'], array( 'percent', 'fixed' ), true )
            ? $_POST['discount_type']
            : 'percent';

        $data['discount_value']     = isset( $_POST['discount_value'] ) ? floatval( $_POST['discount_value'] ) : 0;

        $data['scope']              = isset( $_POST['scope'] ) && in_array( $_POST['scope'], array( 'global', 'categories', 'products' ), true )
            ? $_POST['scope']
            : 'global';

        $data['categories']         = isset( $_POST['categories'] ) ? array_map( 'intval', (array) $_POST['categories'] ) : array();
        $data['products']           = isset( $_POST['products'] ) ? array_map( 'intval', (array) $_POST['products'] ) : array();

        $data['min_cart_subtotal']  = isset( $_POST['min_cart_subtotal'] ) ? wc_format_decimal( $_POST['min_cart_subtotal'] ) : '';
        $data['max_discounts_per_cart'] = isset( $_POST['max_discounts_per_cart'] ) ? max( 0, intval( $_POST['max_discounts_per_cart'] ) ) : '';

        $data['apply_same_product'] = isset( $_POST['apply_same_product'] ) ? 1 : 0;
        $data['allow_coupon_stack'] = isset( $_POST['allow_coupon_stack'] ) ? 1 : 0;

        $data['schedule_enabled']   = isset( $_POST['schedule_enabled'] ) ? 1 : 0;
        $data['schedule_start']     = isset( $_POST['schedule_start'] ) ? sanitize_text_field( $_POST['schedule_start'] ) : '';
        $data['schedule_end']       = isset( $_POST['schedule_end'] ) ? sanitize_text_field( $_POST['schedule_end'] ) : '';

        $data['allowed_roles']      = isset( $_POST['allowed_roles'] ) ? array_map( 'sanitize_text_field', (array) $_POST['allowed_roles'] ) : array();

        $data['banner_text']        = isset( $_POST['banner_text'] ) ? sanitize_text_field( $_POST['banner_text'] ) : '';
        $data['banner_position']    = isset( $_POST['banner_position'] ) && in_array( $_POST['banner_position'], array( 'before_cart', 'after_cart', 'none' ), true )
            ? $_POST['banner_position']
            : 'before_cart';

        update_option( 'gt_bogo_engine_free_rule', $data );

        wp_redirect( admin_url( 'admin.php?page=gt-bogo-engine&tab=free&saved=1' ) );
        exit;
    }

    /**
     * Render the Free Rule page (inside the main engine screen)
     */
    public function render_page() {

        $rule  = $this->get_rule();
        $roles = wp_roles()->roles;

        // Product categories
        $categories = get_terms( array(
            'taxonomy'   => 'product_cat',
            'hide_empty' => false,
        ) );

        // Products – limited for UI; PRO can later add ajax search etc
        $products = wc_get_products( array(
            'limit'   => 200,
            'status'  => 'publish',
            'orderby' => 'title',
            'order'   => 'ASC',
        ) );

        ?>
        <div class="gt-bogo-card">

            <div class="gt-bogo-card-header">
                <h2>Free Global Rule – Buy X Get Y at % Off</h2>
                <p>Included with the free version of GT BOGO Engine. Configure one strong global rule to boost AOV.</p>
            </div>

            <div class="gt-bogo-card-content">

                <?php if ( isset( $_GET['saved'] ) ): ?>
                    <div class="notice notice-success inline"><p>Free rule saved successfully.</p></div>
                <?php endif; ?>

                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" class="gt-bogo-form">
                    <?php wp_nonce_field( 'gt_bogo_save_free_rule_nonce' ); ?>
                    <input type="hidden" name="action" value="gt_bogo_save_free_rule" />

                    <!-- STATUS -->
                    <div class="gt-bogo-setting-row gt-bogo-setting-row--border">
                        <div>
                            <label class="gt-bogo-setting-label">Enable Free Rule</label>
                            <p class="gt-bogo-setting-help">
                                When enabled, this rule will automatically apply in the cart and checkout whenever its conditions are met.
                            </p>
                        </div>
                        <label class="gt-bogo-toggle">
                            <input type="checkbox" name="enabled" <?php checked( $rule['enabled'], 1 ); ?> />
                            <span class="slider"></span>
                        </label>
                    </div>

                    <!-- TITLE -->
                    <div class="gt-bogo-setting-row">
                        <div class="gt-bogo-field-full">
                            <label class="gt-bogo-setting-label" for="gt-bogo-title">Internal Rule Name</label>
                            <input type="text" id="gt-bogo-title" name="title"
                                   value="<?php echo esc_attr( $rule['title'] ); ?>"
                                   class="gt-bogo-input-large" />
                            <p class="gt-bogo-setting-help">
                                Shown only in the admin interface. Use something descriptive like “Global Buy 1 Get 1 50%”.
                            </p>
                        </div>
                    </div>

                    <!-- CORE LOGIC: BUY / GET / DISCOUNT -->
                    <div class="gt-bogo-setting-row gt-bogo-setting-row--columns">
                        <div class="gt-bogo-field">
                            <label class="gt-bogo-setting-label" for="gt-bogo-buy-qty">Buy Quantity (X)</label>
                            <input type="number" min="1" id="gt-bogo-buy-qty" name="buy_qty"
                                   value="<?php echo esc_attr( $rule['buy_qty'] ); ?>" class="small-text" />
                        </div>

                        <div class="gt-bogo-field">
                            <label class="gt-bogo-setting-label" for="gt-bogo-get-qty">Get Quantity (Y)</label>
                            <input type="number" min="1" id="gt-bogo-get-qty" name="get_qty"
                                   value="<?php echo esc_attr( $rule['get_qty'] ); ?>" class="small-text" />
                        </div>

                        <div class="gt-bogo-field">
                            <label class="gt-bogo-setting-label">Discount Type</label>
                            <select name="discount_type" class="gt-bogo-select">
                                <option value="percent" <?php selected( $rule['discount_type'], 'percent' ); ?>>Percentage (%)</option>
                                <option value="fixed"   <?php selected( $rule['discount_type'], 'fixed' ); ?>>Fixed Amount</option>
                            </select>
                        </div>

                        <div class="gt-bogo-field">
                            <label class="gt-bogo-setting-label" for="gt-bogo-discount-value">Discount Value</label>
                            <input type="number" step="0.01" min="0" id="gt-bogo-discount-value" name="discount_value"
                                   value="<?php echo esc_attr( $rule['discount_value'] ); ?>" class="small-text" />
                            <p class="gt-bogo-setting-help">
                                If percentage: 50 = 50% off. If fixed: 10 = 10 in your store currency.
                            </p>
                        </div>
                    </div>

                    <!-- SCOPE -->
                    <div class="gt-bogo-setting-row gt-bogo-setting-row--border">
                        <div class="gt-bogo-field-full">
                            <label class="gt-bogo-setting-label">Scope</label>

                            <div class="gt-bogo-scope-options">
                                <label class="gt-bogo-radio">
                                    <input type="radio" name="scope" value="global" <?php checked( $rule['scope'], 'global' ); ?> />
                                    <span>Global – apply to all products</span>
                                </label>

                                <label class="gt-bogo-radio">
                                    <input type="radio" name="scope" value="categories" <?php checked( $rule['scope'], 'categories' ); ?> />
                                    <span>Selected Categories</span>
                                </label>

                                <label class="gt-bogo-radio">
                                    <input type="radio" name="scope" value="products" <?php checked( $rule['scope'], 'products' ); ?> />
                                    <span>Selected Products</span>
                                </label>
                            </div>

                            <p class="gt-bogo-setting-help">
                                Use categories or individual products if you want to protect margins on specific items.
                            </p>
                        </div>
                    </div>

                    <!-- CATEGORY / PRODUCT PICKERS -->
                    <div class="gt-bogo-setting-row gt-bogo-scope-conditional gt-bogo-scope-categories">
                        <div class="gt-bogo-field-full">
                            <label class="gt-bogo-setting-label">Eligible Categories</label>
                            <select name="categories[]" multiple="multiple" class="gt-bogo-multiselect">
                                <?php if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) : ?>
                                    <?php foreach ( $categories as $cat ) : ?>
                                        <option value="<?php echo esc_attr( $cat->term_id ); ?>"
                                                <?php selected( in_array( $cat->term_id, $rule['categories'], true ), true ); ?>>
                                            <?php echo esc_html( $cat->name ); ?>
                                        </option>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </select>
                            <p class="gt-bogo-setting-help">
                                BOGO discounts will only trigger when the cart contains products from these categories.
                            </p>
                        </div>
                    </div>

                    <div class="gt-bogo-setting-row gt-bogo-scope-conditional gt-bogo-scope-products">
                        <div class="gt-bogo-field-full">
                            <label class="gt-bogo-setting-label">Eligible Products</label>
                            <select name="products[]" multiple="multiple" class="gt-bogo-multiselect">
                                <?php if ( ! empty( $products ) ) : ?>
                                    <?php foreach ( $products as $product ) : ?>
                                        <option value="<?php echo esc_attr( $product->get_id() ); ?>"
                                                <?php selected( in_array( $product->get_id(), $rule['products'], true ), true ); ?>>
                                            <?php echo esc_html( $product->get_name() ); ?> (<?php echo esc_html( $product->get_sku() ); ?>)
                                        </option>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </select>
                            <p class="gt-bogo-setting-help">
                                BOGO discounts will only trigger on these products.
                            </p>
                        </div>
                    </div>

                    <!-- CONDITIONS -->
                    <div class="gt-bogo-setting-row gt-bogo-setting-row--border">
                        <div class="gt-bogo-field">
                            <label class="gt-bogo-setting-label" for="gt-bogo-min-subtotal">Minimum Cart Subtotal</label>
                            <input type="number" step="0.01" min="0" id="gt-bogo-min-subtotal" name="min_cart_subtotal"
                                   value="<?php echo esc_attr( $rule['min_cart_subtotal'] ); ?>" class="small-text" />
                            <p class="gt-bogo-setting-help">
                                Optional. Example: 50 means cart must reach $50 before BOGO applies.
                            </p>
                        </div>

                        <div class="gt-bogo-field">
                            <label class="gt-bogo-setting-label" for="gt-bogo-max-discounts">Maximum Discounted Items Per Cart</label>
                            <input type="number" min="0" id="gt-bogo-max-discounts" name="max_discounts_per_cart"
                                   value="<?php echo esc_attr( $rule['max_discounts_per_cart'] ); ?>" class="small-text" />
                            <p class="gt-bogo-setting-help">
                                Optional cap. 0 = no limit.
                            </p>
                        </div>
                    </div>

                    <!-- FLAGS -->
                    <div class="gt-bogo-setting-row">
                        <div class="gt-bogo-field-full gt-bogo-flags">
                            <label class="gt-bogo-checkbox">
                                <input type="checkbox" name="apply_same_product" <?php checked( $rule['apply_same_product'], 1 ); ?> />
                                <span>Apply discount only when X and Y are the same product</span>
                            </label>

                            <label class="gt-bogo-checkbox">
                                <input type="checkbox" name="allow_coupon_stack" <?php checked( $rule['allow_coupon_stack'], 1 ); ?> />
                                <span>Allow this BOGO to stack with coupons</span>
                            </label>

                            <p class="gt-bogo-setting-help">
                                For most stores, we recommend keeping stacking disabled to protect margins.
                            </p>
                        </div>
                    </div>

                    <!-- SCHEDULING -->
                    <div class="gt-bogo-setting-row gt-bogo-setting-row--border">
                        <div class="gt-bogo-field-full">
                            <label class="gt-bogo-setting-label">Schedule</label>

                            <label class="gt-bogo-checkbox">
                                <input type="checkbox" name="schedule_enabled" <?php checked( $rule['schedule_enabled'], 1 ); ?> />
                                <span>Enable schedule for this rule</span>
                            </label>

                            <div class="gt-bogo-inline-fields">
                                <div class="gt-bogo-field">
                                    <label for="gt-bogo-schedule-start">Start Date (optional)</label>
                                    <input type="date" id="gt-bogo-schedule-start" name="schedule_start"
                                           value="<?php echo esc_attr( $rule['schedule_start'] ); ?>" />
                                </div>

                                <div class="gt-bogo-field">
                                    <label for="gt-bogo-schedule-end">End Date (optional)</label>
                                    <input type="date" id="gt-bogo-schedule-end" name="schedule_end"
                                           value="<?php echo esc_attr( $rule['schedule_end'] ); ?>" />
                                </div>
                            </div>

                            <p class="gt-bogo-setting-help">
                                Leave both blank to run this rule indefinitely.
                            </p>
                        </div>
                    </div>

                    <!-- ROLES -->
                    <div class="gt-bogo-setting-row">
                        <div class="gt-bogo-field-full">
                            <label class="gt-bogo-setting-label">Eligible User Roles</label>
                            <select name="allowed_roles[]" multiple="multiple" class="gt-bogo-multiselect">
                                <?php foreach ( $roles as $role_key => $role_data ) : ?>
                                    <option value="<?php echo esc_attr( $role_key ); ?>"
                                            <?php selected( in_array( $role_key, $rule['allowed_roles'], true ), true ); ?>>
                                        <?php echo esc_html( $role_data['name'] ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <p class="gt-bogo-setting-help">
                                Leave empty to allow all customers, including guests.
                            </p>
                        </div>
                    </div>

                    <!-- PROMO BANNER -->
                    <div class="gt-bogo-setting-row gt-bogo-setting-row--border">
                        <div class="gt-bogo-field-full">
                            <label class="gt-bogo-setting-label" for="gt-bogo-banner-text">Cart Promotion Banner</label>
                            <input type="text" id="gt-bogo-banner-text" name="banner_text"
                                   value="<?php echo esc_attr( $rule['banner_text'] ); ?>"
                                   class="gt-bogo-input-large" />
                            <p class="gt-bogo-setting-help">
                                Short message shown in the cart when this rule is active. Helps convert more customers.
                            </p>

                            <label class="gt-bogo-setting-label">Banner Position</label>
                            <select name="banner_position" class="gt-bogo-select">
                                <option value="before_cart" <?php selected( $rule['banner_position'], 'before_cart' ); ?>>Before cart totals</option>
                                <option value="after_cart"  <?php selected( $rule['banner_position'], 'after_cart' ); ?>>After cart totals</option>
                                <option value="none"        <?php selected( $rule['banner_position'], 'none' ); ?>>Do not show banner</option>
                            </select>
                        </div>
                    </div>

                    <!-- SUBMIT -->
                    <p>
                        <button type="submit" class="button button-primary button-large">
                            Save Free Rule
                        </button>
                    </p>

                </form>

            </div>
        </div>
        <?php
    }
}

GT_BOGO_Engine_Free_Rule::instance();

