<?php
/**
 * GT BOGO Rules Engine – Settings Page Controller
 * Provides premium-grade UI for global settings, debug tools, and PRO upgrades.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class GT_BOGO_Engine_Settings {

    private static $instance = null;

    /**
     * Singleton instance
     */
    public static function instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action( 'admin_menu', array( $this, 'register_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_post_gt_bogo_save_settings', array( $this, 'save_settings' ) );
        add_action( 'wp_ajax_gt_bogo_clear_log', array( $this, 'ajax_clear_log' ) );
    }

    /**
     * Register submenu under WooCommerce
     */
    public function register_menu() {

        add_submenu_page(
            'woocommerce',
            'GT BOGO Engine Settings',
            'BOGO Engine',
            'manage_woocommerce',
            'gt-bogo-settings',
            array( $this, 'render_settings_page' )
        );
    }

    /**
     * Register stored settings
     */
    public function register_settings() {

        register_setting( 'gt_bogo_engine_settings_group', 'gt_bogo_engine_settings' );
        register_setting( 'gt_bogo_engine_debug_group', 'gt_bogo_engine_debug_log' );
    }

    /**
     * Save settings
     */
    public function save_settings() {

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( 'Not allowed.' );
        }

        check_admin_referer( 'gt_bogo_save_settings_nonce' );

        $settings = array(
            'enable_engine'  => isset( $_POST['enable_engine'] ) ? 1 : 0,
            'debug_mode'     => isset( $_POST['debug_mode'] ) ? 1 : 0,
            'pro_nag_hidden' => 0,
        );

        update_option( 'gt_bogo_engine_settings', $settings );

        wp_redirect( admin_url( 'admin.php?page=gt-bogo-settings&saved=1' ) );
        exit;
    }

    /**
     * AJAX clear debug log
     */
    public function ajax_clear_log() {

        update_option( 'gt_bogo_engine_debug_log', '' );
        wp_send_json_success();
    }

    /**
     * Render Settings Page
     */
    public function render_settings_page() {

        $settings = get_option( 'gt_bogo_engine_settings', array(
            'enable_engine'  => 1,
            'debug_mode'     => 0,
            'pro_nag_hidden' => 0,
        ) );

        $debug_log = get_option( 'gt_bogo_engine_debug_log', '' );

        ?>
        <div class="wrap gt-bogo-wrap">

            <h1 class="gt-bogo-title">GT BOGO Engine – Settings</h1>

            <?php if ( isset( $_GET['saved'] ) ): ?>
                <div class="notice notice-success"><p>Settings saved.</p></div>
            <?php endif; ?>

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'gt_bogo_save_settings_nonce' ); ?>
                <input type="hidden" name="action" value="gt_bogo_save_settings">

                <!-- SETTINGS CARD -->
                <div class="gt-bogo-card">
                    <div class="gt-bogo-card-header">
                        <h2>General Settings</h2>
                    </div>

                    <div class="gt-bogo-card-content">

                        <!-- Engine Toggle -->
                        <div class="gt-bogo-setting-row">
                            <label class="gt-bogo-setting-label">Enable BOGO Engine</label>
                            <label class="gt-bogo-toggle">
                                <input type="checkbox" name="enable_engine" <?php checked( $settings['enable_engine'], 1 ); ?>>
                                <span class="slider"></span>
                            </label>
                        </div>

                        <!-- Debug Mode -->
                        <div class="gt-bogo-setting-row">
                            <label class="gt-bogo-setting-label">Enable Debug Mode</label>
                            <label class="gt-bogo-toggle">
                                <input type="checkbox" name="debug_mode" <?php checked( $settings['debug_mode'], 1 ); ?>>
                                <span class="slider"></span>
                            </label>
                        </div>

                    </div>
                </div>

                <p>
                    <button class="button button-primary button-large">Save Settings</button>
                </p>

            </form>

            <!-- DEBUG LOG SECTION -->
            <div class="gt-bogo-card">
                <div class="gt-bogo-card-header">
                    <h2>Debug Log</h2>
                </div>

                <div class="gt-bogo-card-content">
                    <pre class="gt-bogo-log-view"><?php echo esc_html( $debug_log ?: 'Log is empty.' ); ?></pre>

                    <button id="gt-bogo-clear-log" class="button">Clear Log</button>
                </div>
            </div>

            <!-- PRO UPGRADE CARD -->
            <div class="gt-bogo-card">
                <div class="gt-bogo-card-header">
                    <h2>Upgrade to PRO</h2>
                </div>

                <div class="gt-bogo-card-content">
                    <p>Unlock advanced rule types, multiple rule sets, cheapest-item logic, scheduling, and more.</p>

                    <a href="https://graphictshirts.shop/bogo" target="_blank" class="button button-primary button-large">
                        Get PRO – $19.99 / year
                    </a>
                </div>
            </div>

            <!-- PRO MODAL -->
            <div class="gt-bogo-modal-backdrop" data-modal="pro-upgrade">
                <div class="gt-bogo-modal">
                    <button class="gt-bogo-modal-close">×</button>
                    <h3>Upgrade Required</h3>
                    <p>This feature is part of GT BOGO Engine PRO.</p>
                    <a href="https://graphictshirts.shop/bogo" target="_blank" class="gt-bogo-modal-btn">Upgrade Now</a>
                </div>
            </div>

            <!-- CLEAR LOG CONFIRMATION MODAL -->
            <div class="gt-bogo-modal-backdrop" data-modal="clear-log-confirm">
                <div class="gt-bogo-modal">
                    <button class="gt-bogo-modal-close">×</button>
                    <h3>Clear Debug Log?</h3>
                    <p>This will permanently remove the log contents.</p>
                    <button id="gt-bogo-confirm-clear-log" class="gt-bogo-modal-btn">Clear Log</button>
                </div>
            </div>

        </div>
        <?php
    }
}

GT_BOGO_Engine_Settings::instance();

