<?php
/**
 * GT BOGO Rules Engine – Cart Logic
 *
 * Applies BOGO discounts in WooCommerce cart & checkout.
 * Handles:
 *  - Free core rule: Buy X Get Y % Off (global / categories / products)
 *  - Architecture for future PRO rule types
 *
 * @package GT_BOGO_Engine
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Main cart engine for GT BOGO.
 */
class GT_BOGO_Engine_Cart {

    /**
     * Singleton instance.
     *
     * @var GT_BOGO_Engine_Cart|null
     */
    protected static $instance = null;

    /**
     * Get singleton instance.
     *
     * @return GT_BOGO_Engine_Cart
     */
    public static function instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     */
    private function __construct() {
        // Core hook for adjusting prices.
        add_action( 'woocommerce_before_calculate_totals', array( $this, 'apply_discounts' ), 20, 1 );

        // Optional: show small label in cart items.
        add_filter( 'woocommerce_cart_item_price', array( $this, 'maybe_render_bogo_label_in_price' ), 10, 3 );
    }

    /**
     * Get plugin settings.
     *
     * @return array
     */
    protected function get_settings() {
        $settings = get_option( 'gt_bogo_engine_settings', array() );

        $defaults = array(
            'enable_engine'  => 1,
            'debug_mode'     => 0,
            'pro_nag_hidden' => 0,
        );

        return wp_parse_args( $settings, $defaults );
    }

    /**
     * Get free core rule definition.
     *
     * @return array|null
     */
    protected function get_free_rule() {
        $rule = get_option( 'gt_bogo_engine_free_rule', array() );

        if ( empty( $rule ) || empty( $rule['enabled'] ) ) {
            return null;
        }

        // Ensure all expected fields exist.
        $defaults = array(
            'enabled'        => 0,
            'title'          => 'Buy 1 Get 1 at 50% Off (Global)',
            'buy_qty'        => 1,
            'get_qty'        => 1,
            'discount_type'  => 'percent',
            'discount_value' => 50,
            'scope'          => 'global',   // global|categories|products
            'categories'     => array(),    // category IDs.
            'products'       => array(),    // product IDs.
        );

        $rule = wp_parse_args( $rule, $defaults );

        // Sanitize numeric fields.
        $rule['buy_qty']        = max( 1, absint( $rule['buy_qty'] ) );
        $rule['get_qty']        = max( 1, absint( $rule['get_qty'] ) );
        $rule['discount_value'] = max( 0, min( 100, floatval( $rule['discount_value'] ) ) );

        return $rule;
    }

    /**
     * Main hook: apply BOGO discounts to the cart.
     *
     * @param WC_Cart $cart WooCommerce cart object.
     */
    public function apply_discounts( $cart ) {
        if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
            return;
        }

        if ( ! $cart || $cart->is_empty() ) {
            return;
        }

        $settings = $this->get_settings();
        if ( empty( $settings['enable_engine'] ) ) {
            // Engine disabled in settings.
            return;
        }

        // Reset any previous BOGO adjustments.
        $this->reset_bogo_adjustments( $cart );

        // Apply free core rule.
        $free_rule = $this->get_free_rule();
        if ( $free_rule ) {
            $this->apply_free_percent_rule( $cart, $free_rule );
        }

        // Architecture for PRO rules (future expansion).
        // $pro_rules = get_option( 'gt_bogo_engine_rules', array() );
        // if ( is_array( $pro_rules ) && ! empty( $pro_rules ) ) {
        //     foreach ( $pro_rules as $rule_id => $rule ) {
        //         // Here we would call type-specific handlers, e.g.:
        //         // $this->apply_pro_rule( $cart, $rule );
        //     }
        // }

        // All discounts are applied by price adjustments, Woo will handle totals.
    }

    /**
     * Reset line item prices back to their base (pre-BOGO) values.
     *
     * We store the original unit price in `gt_bogo_base_price`
     * to avoid compounding discounts when Woo recalculates totals multiple times.
     *
     * @param WC_Cart $cart Cart.
     */
    protected function reset_bogo_adjustments( $cart ) {
        foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {

            // If we previously stored a base price, restore it.
            if ( isset( $cart_item['gt_bogo_base_price'] ) ) {
                $base_price = floatval( $cart_item['gt_bogo_base_price'] );
                $cart_item['data']->set_price( $base_price );
                $cart->cart_contents[ $cart_item_key ]['data'] = $cart_item['data'];
            } else {
                // First run: store the current price as the base.
                $base_price = floatval( $cart_item['data']->get_price() );
                $cart->cart_contents[ $cart_item_key ]['gt_bogo_base_price'] = $base_price;
            }

            // Clear any previous BOGO flags.
            unset( $cart->cart_contents[ $cart_item_key ]['gt_bogo_discounted_units'] );
            unset( $cart->cart_contents[ $cart_item_key ]['gt_bogo_discount_amount'] );
            unset( $cart->cart_contents[ $cart_item_key ]['gt_bogo_label'] );
        }
    }

    /**
     * Check if a cart item is eligible for a given rule based on scope & targeting.
     *
     * @param array $cart_item Cart item.
     * @param array $rule      Rule definition.
     *
     * @return bool
     */
    protected function is_item_eligible_for_rule( $cart_item, $rule ) {
        $product = $cart_item['data'];

        if ( ! $product || ! is_a( $product, 'WC_Product' ) ) {
            return false;
        }

        $scope = isset( $rule['scope'] ) ? $rule['scope'] : 'global';

        // Global: all products eligible.
        if ( $scope === 'global' ) {
            return true;
        }

        $product_id = $product->get_id();

        // Products-based scope.
        if ( $scope === 'products' && ! empty( $rule['products'] ) ) {
            $ids = array_map( 'absint', (array) $rule['products'] );
            return in_array( $product_id, $ids, true );
        }

        // Categories-based scope.
        if ( $scope === 'categories' && ! empty( $rule['categories'] ) ) {
            $category_ids = array_map( 'absint', (array) $rule['categories'] );

            $product_terms = wc_get_product_term_ids( $product_id, 'product_cat' );
            if ( empty( $product_terms ) ) {
                return false;
            }

            return (bool) array_intersect( $category_ids, $product_terms );
        }

        return false;
    }

    /**
     * Free rule handler: Buy X Get Y % off on eligible items.
     *
     * Logic:
     *  - Count total eligible quantity across all items.
     *  - Determine number of discounted units: groups * get_qty, where
     *    groups = floor( total_qty / (buy_qty + get_qty) )
     *  - Discount the cheapest eligible units.
     *
     * @param WC_Cart $cart Cart.
     * @param array   $rule Rule definition.
     */
    protected function apply_free_percent_rule( $cart, $rule ) {
        $buy_qty        = max( 1, absint( $rule['buy_qty'] ) );
        $get_qty        = max( 1, absint( $rule['get_qty'] ) );
        $discount_type  = $rule['discount_type'];
        $discount_value = floatval( $rule['discount_value'] );

        if ( $discount_type !== 'percent' || $discount_value <= 0 ) {
            return;
        }

        // STEP 1: Build list of eligible units across all cart lines.
        $eligible_units = array(); // Each entry: [ 'key' => cart_item_key, 'price' => unit_price ].
        $total_qty      = 0;

        foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {

            if ( ! $this->is_item_eligible_for_rule( $cart_item, $rule ) ) {
                continue;
            }

            $qty = (int) $cart_item['quantity'];
            if ( $qty <= 0 ) {
                continue;
            }

            // Use stored base price to avoid compounding discounts.
            $base_price = isset( $cart_item['gt_bogo_base_price'] )
                ? floatval( $cart_item['gt_bogo_base_price'] )
                : floatval( $cart_item['data']->get_price() );

            if ( $base_price <= 0 ) {
                continue;
            }

            for ( $i = 0; $i < $qty; $i++ ) {
                $eligible_units[] = array(
                    'key'   => $cart_item_key,
                    'price' => $base_price,
                );
            }

            $total_qty += $qty;
        }

        if ( $total_qty <= 0 || empty( $eligible_units ) ) {
            return;
        }

        // STEP 2: Determine how many units should be discounted.
        $group_size        = $buy_qty + $get_qty;
        $groups            = floor( $total_qty / $group_size );
        $discounted_units  = $groups * $get_qty;

        if ( $discounted_units <= 0 ) {
            return;
        }

        // STEP 3: Sort units by price (cheapest first).
        usort(
            $eligible_units,
            function ( $a, $b ) {
                if ( $a['price'] === $b['price'] ) {
                    return 0;
                }
                return ( $a['price'] < $b['price'] ) ? -1 : 1;
            }
        );

        // STEP 4: Determine discount per cart item.
        $discount_map = array(); // key => [ 'units' => int, 'amount' => float ].

        $units_to_discount = array_slice( $eligible_units, 0, $discounted_units );

        foreach ( $units_to_discount as $unit ) {
            $key          = $unit['key'];
            $unit_price   = floatval( $unit['price'] );
            $unit_discount = ( $discount_value / 100 ) * $unit_price;

            if ( ! isset( $discount_map[ $key ] ) ) {
                $discount_map[ $key ] = array(
                    'units'  => 0,
                    'amount' => 0.0,
                );
            }

            $discount_map[ $key ]['units']  += 1;
            $discount_map[ $key ]['amount'] += $unit_discount;
        }

        if ( empty( $discount_map ) ) {
            return;
        }

        // STEP 5: Apply adjusted prices to cart items.
        foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {

            if ( ! isset( $discount_map[ $cart_item_key ] ) ) {
                continue;
            }

            $line_discount_units = $discount_map[ $cart_item_key ]['units'];
            $line_discount_amount = $discount_map[ $cart_item_key ]['amount'];

            $qty = (int) $cart_item['quantity'];
            if ( $qty <= 0 ) {
                continue;
            }

            $base_price = isset( $cart_item['gt_bogo_base_price'] )
                ? floatval( $cart_item['gt_bogo_base_price'] )
                : floatval( $cart_item['data']->get_price() );

            if ( $base_price <= 0 ) {
                continue;
            }

            // Per-unit discount spread across this line.
            $per_unit_discount = $line_discount_amount / $qty;

            // Cap discount so we never go negative.
            $new_price = max( 0, $base_price - $per_unit_discount );

            $cart->cart_contents[ $cart_item_key ]['data']->set_price( $new_price );

            // Store metadata for display/debug.
            $cart->cart_contents[ $cart_item_key ]['gt_bogo_discounted_units'] = $line_discount_units;
            $cart->cart_contents[ $cart_item_key ]['gt_bogo_discount_amount']  = wc_format_decimal( $line_discount_amount );
            $cart->cart_contents[ $cart_item_key ]['gt_bogo_label']            = sprintf(
                /* translators: 1: discount value, 2: rule title */
                __( 'BOGO: %1$s%% off (%2$s)', 'gt-bogo-engine' ),
                $discount_value,
                isset( $rule['title'] ) ? $rule['title'] : __( 'BOGO Offer', 'gt-bogo-engine' )
            );
        }
    }

    /**
     * Render a small label under the price if BOGO is applied.
     *
     * @param string $price_html Price HTML.
     * @param array  $cart_item  Cart item.
     * @param string $cart_item_key Cart item key.
     *
     * @return string
     */
    public function maybe_render_bogo_label_in_price( $price_html, $cart_item, $cart_item_key ) {

        if ( isset( $cart_item['gt_bogo_label'] ) && ! empty( $cart_item['gt_bogo_label'] ) ) {
            $label = esc_html( $cart_item['gt_bogo_label'] );
            $price_html .= sprintf(
                '<br/><small class="gt-bogo-price-label" style="color:#059669;font-size:11px;">%s</small>',
                $label
            );
        }

        return $price_html;
    }
}

GT_BOGO_Engine_Cart::instance();
