<?php
/**
 * GT BOGO ENGINE — Admin Backend Controller
 *
 * Controls:
 * - Admin menu & subpages
 * - Sidebar UI
 * - Page routing
 * - Admin assets enqueue
 * - Autosave (AJAX)
 * - Import/Export
 * - Welcome page
 * - Documentation page
 *
 * @package GT_BOGO_Engine
 */

if (!defined('ABSPATH')) {
    exit;
}

class GT_BOGO_Engine_Admin {

    /**
     * Constructor
     */
    public function __construct() {

        // Admin pages
        add_action('admin_menu', array($this, 'register_menu'));

        // Assets
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));

        // AJAX handlers
        add_action('wp_ajax_gt_bogo_engine_autosave', array($this, 'ajax_autosave'));
        add_action('wp_ajax_gt_bogo_engine_export_rules', array($this, 'ajax_export_rules'));
        add_action('wp_ajax_gt_bogo_engine_import_rules', array($this, 'ajax_import_rules'));
    }

    /**
     * Register admin menu
     */
    public function register_menu() {

        add_menu_page(
            __('GT BOGO ENGINE', GT_BOGO_ENGINE_MENU_SLUG),
            __('GT BOGO ENGINE', GT_BOGO_ENGINE_MENU_SLUG),
            'manage_options',
            GT_BOGO_ENGINE_MENU_SLUG,
            array($this, 'render_page'),
            'dashicons-tag',
            57
        );
    }

    /**
     * Sidebar (left UI)
     */
    private function render_sidebar() {

        $active = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'dashboard';
        ?>
        <div class="gt-bogo-engine-sidebar">

            <div class="gt-bogo-brand">
                <img src="<?php echo esc_url(GT_BOGO_ENGINE_URL . 'assets/icon-128x128.png'); ?>" alt="GT BOGO ENGINE" />
                <h3>GT BOGO ENGINE</h3>
            </div>

            <ul class="gt-bogo-menu">
                <li class="<?php echo $active === 'dashboard' ? 'active' : ''; ?>">
                    <a href="?page=gt-bogo-engine&tab=dashboard"><?php _e('Dashboard', GT_BOGO_ENGINE_MENU_SLUG); ?></a>
                </li>
                <li class="<?php echo $active === 'settings' ? 'active' : ''; ?>">
                    <a href="?page=gt-bogo-engine&tab=settings"><?php _e('Settings', GT_BOGO_ENGINE_MENU_SLUG); ?></a>
                </li>
                <li class="<?php echo $active === 'free-rule' ? 'active' : ''; ?>">
                    <a href="?page=gt-bogo-engine&tab=free-rule"><?php _e('Free Core Rule', GT_BOGO_ENGINE_MENU_SLUG); ?></a>
                </li>
                <li class="<?php echo $active === 'pro-rules' ? 'active' : ''; ?>">
                    <a href="?page=gt-bogo-engine&tab=pro-rules"><?php _e('PRO Rules', GT_BOGO_ENGINE_MENU_SLUG); ?></a>
                </li>
                <li class="<?php echo $active === 'pro-packs' ? 'active' : ''; ?>">
                    <a href="?page=gt-bogo-engine&tab=pro-packs"><?php _e('PRO Packs', GT_BOGO_ENGINE_MENU_SLUG); ?></a>
                </li>
                <li class="<?php echo $active === 'import-export' ? 'active' : ''; ?>">
                    <a href="?page=gt-bogo-engine&tab=import-export"><?php _e('Import / Export', GT_BOGO_ENGINE_MENU_SLUG); ?></a>
                </li>
                <li class="<?php echo $active === 'documentation' ? 'active' : ''; ?>">
                    <a href="?page=gt-bogo-engine&tab=documentation"><?php _e('Documentation', GT_BOGO_ENGINE_MENU_SLUG); ?></a>
                </li>
                <li class="<?php echo $active === 'welcome' ? 'active' : ''; ?>">
                    <a href="?page=gt-bogo-engine&tab=welcome" class="welcome-link"><?php _e('Welcome', GT_BOGO_ENGINE_MENU_SLUG); ?></a>
                </li>
            </ul>

            <div class="gt-bogo-pro-box">
                <h4>GT BOGO PRO</h4>
                <p><?php _e('Unlock deeper discount logic, stackable rules, scheduling, cart conditions & more.', GT_BOGO_ENGINE_MENU_SLUG); ?></p>
                <a target="_blank" href="https://graphictshirts.shop/bogo" class="gt-bogo-pro-btn">Upgrade to PRO</a>
            </div>

        </div>
        <?php
    }

    /**
     * Page renderer (main content area)
     */
    public function render_page() {

        $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'dashboard';

        echo '<div class="gt-bogo-engine-wrapper">';

        // Sidebar
        $this->render_sidebar();

        // Content
        echo '<div class="gt-bogo-engine-content">';

        switch ($tab) {
            case 'settings':
                require GT_BOGO_ENGINE_DIR . 'includes/admin/views/page-settings.php';
                break;

            case 'free-rule':
                require GT_BOGO_ENGINE_DIR . 'includes/admin/views/page-free-rule.php';
                break;

            case 'pro-rules':
                require GT_BOGO_ENGINE_DIR . 'includes/admin/views/page-pro-rules.php';
                break;

            case 'pro-packs':
                require GT_BOGO_ENGINE_DIR . 'includes/admin/views/page-pro-packs.php';
                break;

            case 'documentation':
                require GT_BOGO_ENGINE_DIR . 'includes/admin/views/page-documentation.php';
                break;

            case 'import-export':
                require GT_BOGO_ENGINE_DIR . 'includes/admin/views/page-import-export.php';
                break;

            case 'welcome':
                require GT_BOGO_ENGINE_DIR . 'includes/admin/views/page-welcome.php';
                break;

            case 'dashboard':
            default:
                require GT_BOGO_ENGINE_DIR . 'includes/admin/views/page-dashboard.php';
        }

        echo '</div>'; // content
        echo '</div>'; // wrapper
    }

    /**
     * Enqueue premium admin assets
     */
    public function enqueue_assets($hook) {

        if (strpos($hook, GT_BOGO_ENGINE_MENU_SLUG) === false) {
            return;
        }

        wp_enqueue_style(
            'gt-bogo-engine-admin',
            GT_BOGO_ENGINE_URL . 'assets/css/admin.css',
            array(),
            GT_BOGO_ENGINE_VERSION
        );

        wp_enqueue_script(
            'gt-bogo-engine-admin',
            GT_BOGO_ENGINE_URL . 'assets/js/admin.js',
            array('jquery'),
            GT_BOGO_ENGINE_VERSION,
            true
        );

        wp_localize_script(
            'gt-bogo-engine-admin',
            'GT_BOGO_Engine_Admin',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('gt_bogo_engine_nonce'),
            )
        );
    }

    /**
     * AJAX — Autosave (settings, free rule, pro rules)
     */
    public function ajax_autosave() {

        check_ajax_referer('gt_bogo_engine_nonce', 'gt_bogo_engine_nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', GT_BOGO_ENGINE_MENU_SLUG)));
        }

        $context = $_POST['gt_bogo_engine_context'] ?? '';

        switch ($context) {

            case 'settings':
                update_option('gt_bogo_engine_settings', array(
                    'enable_engine'  => isset($_POST['enable_engine']) ? 1 : 0,
                    'debug_mode'     => isset($_POST['debug_mode']) ? 1 : 0,
                    'pro_nag_hidden' => isset($_POST['pro_nag_hidden']) ? 1 : 0,
                ));
                wp_send_json_success(array('message' => 'Settings saved.'));
                break;

            case 'free_rule':
                update_option('gt_bogo_engine_free_rule', array(
                    'enabled'        => isset($_POST['free_rule_enabled']) ? 1 : 0,
                    'title'          => sanitize_text_field($_POST['free_rule_title'] ?? ''),
                    'buy_qty'        => intval($_POST['free_rule_buy_qty'] ?? 1),
                    'get_qty'        => intval($_POST['free_rule_get_qty'] ?? 1),
                    'discount_type'  => sanitize_text_field($_POST['free_rule_discount_type'] ?? 'percent'),
                    'discount_value' => floatval($_POST['free_rule_discount_value'] ?? 0),
                ));
                wp_send_json_success(array('message' => 'Free rule saved.'));
                break;

            case 'pro_rules':
                update_option('gt_bogo_engine_rules', $_POST['rules'] ?? array());
                wp_send_json_success(array('message' => 'PRO rules saved.'));
                break;

            default:
                wp_send_json_error(array('message' => 'Invalid autosave target.'));
        }
    }

    /**
     * AJAX — Export rules
     */
    public function ajax_export_rules() {

        check_ajax_referer('gt_bogo_engine_nonce', 'nonce');

        $data = array(
            'settings'   => get_option('gt_bogo_engine_settings', array()),
            'free_rule'  => get_option('gt_bogo_engine_free_rule', array()),
            'pro_rules'  => get_option('gt_bogo_engine_rules', array()),
        );

        wp_send_json_success($data);
    }

    /**
     * AJAX — Import rules
     */
    public function ajax_import_rules() {

        check_ajax_referer('gt_bogo_engine_nonce', 'nonce');

        $raw = stripslashes($_POST['import_json'] ?? '');
        $json = json_decode($raw, true);

        if (!is_array($json)) {
            wp_send_json_error(array('message' => 'Invalid JSON.'));
        }

        if (isset($json['settings'])) {
            update_option('gt_bogo_engine_settings', $json['settings']);
        }

        if (isset($json['free_rule'])) {
            update_option('gt_bogo_engine_free_rule', $json['free_rule']);
        }

        if (isset($json['pro_rules'])) {
            update_option('gt_bogo_engine_rules', $json['pro_rules']);
        }

        wp_send_json_success(array('message' => 'Rules imported successfully.'));
    }


    /**
     * Inject small CSS tweaks just for GT BOGO ENGINE admin pages.
     */
    public function admin_head_css() {
        $screen = get_current_screen();
        if ( empty( $screen ) ) {
            return;
        }

        if ( strpos( $screen->id, GT_BOGO_ENGINE_MENU_SLUG ) === false ) {
            return;
        }
        ?>
        <style>
            .wrap > img:first-of-type {
                display: none !important;
            }
        </style>
        <?php
    }

}
