<?php
/**
 * GT BOGO Engine – Cart Logic
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class GT_BOGO_Engine_Cart {

    public function __construct() {
        add_action( 'woocommerce_before_calculate_totals', array( $this, 'apply_discounts' ), 25 );
    }

    /**
     * Main entry point
     */
    public function apply_discounts( $cart ) {

        if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
            return;
        }

        if ( empty( $cart->get_cart() ) ) {
            return;
        }

        $settings = get_option( 'gt_bogo_engine_settings', array() );
        if ( empty( $settings['enable_engine'] ) ) {
            return;
        }

        $free_rule = get_option( 'gt_bogo_engine_free_rule', array() );
        $pro_rules = get_option( 'gt_bogo_engine_rules', array() );
        // PRO rules require an active license. If not valid, ignore stored PRO rules.
        if ( ! empty( $pro_rules ) && class_exists( 'GT_BOGO_Engine_License' ) ) {
            $lic = GT_BOGO_Engine_License::instance();
            if ( ! $lic->is_pro_active() ) {
                $pro_rules = array();
            }
        }

        // Reset previous discounts
        foreach ( $cart->get_cart() as $cart_item_key => $item ) {
            $item['gt_bogo_discount'] = 0;
        }

        /**
         * Apply Free Rule (global starter rule)
         */
        if ( ! empty( $free_rule['enabled'] ) ) {
            $this->apply_rule( $cart, $free_rule, 'free' );
        }

        /**
         * Apply PRO Rules
         */
        if ( ! empty( $pro_rules ) ) {
            foreach ( $pro_rules as $rule ) {
                if ( ! empty( $rule['enabled'] ) ) {
                    $this->apply_rule( $cart, $rule, 'pro' );
                }
            }
        }

    }

    /**
     * Applies one BOGO rule
     */
    private function apply_rule( $cart, $rule, $type ) {

        $buy_qty = intval( $rule['buy_qty'] );
        $get_qty = intval( $rule['get_qty'] );
        $discount_type  = $rule['discount_type'];
        $discount_value = floatval( $rule['discount_value'] );

        if ( $buy_qty <= 0 || $get_qty <= 0 ) {
            return;
        }

        // Build a list of matching products in cart
        $cart_items = array();

        foreach ( $cart->get_cart() as $key => $item ) {

            $product_id = $item['product_id'];

            // Scope: global / category / product_id
            if ( $rule['scope'] === 'categories' ) {
                if ( ! has_term( $rule['categories'], 'product_cat', $product_id ) ) {
                    continue;
                }
            }

            if ( $rule['scope'] === 'products' ) {
                if ( ! in_array( $product_id, $rule['products'] ) ) {
                    continue;
                }
            }

            $cart_items[$key] = array(
                'qty'  => $item['quantity'],
                'price'=> $item['data']->get_price(),
            );
        }

        if ( empty( $cart_items ) ) {
            return;
        }

        /**
         * Determine how many “get” items we can discount.
         */
        $total_buy_units = 0;
        foreach ( $cart_items as $it ) {
            $total_buy_units += $it['qty'];
        }

        $eligible_sets = floor( $total_buy_units / $buy_qty );
        $max_get_items = $eligible_sets * $get_qty;

        if ( $max_get_items <= 0 ) {
            return;
        }

        /**
         * Apply discount to the cheapest eligible products.
         */
        uasort( $cart_items, function( $a, $b ) {
            return $a['price'] <=> $b['price'];
        });

        foreach ( $cart_items as $key => $cart_item ) {

            if ( $max_get_items <= 0 ) {
                break;
            }

            $line_qty = $cart_item['qty'];

            for ( $i = 0; $i < $line_qty && $max_get_items > 0; $i++ ) {

                $price = $cart_item['price'];

                switch ( $discount_type ) {
                    case 'percent':
                        $discount_amount = $price * ( $discount_value / 100 );
                        break;

                    case 'fixed':
                        $discount_amount = min( $price, $discount_value );
                        break;

                    default:
                        $discount_amount = 0;
                }

                // Apply discount to item
                $item = $cart->get_cart_item( $key );
                $item['gt_bogo_discount'] = ( $item['gt_bogo_discount'] ?? 0 ) + $discount_amount;

                $max_get_items--;
            }
        }

        /**
         * Apply actual price reductions to WooCommerce
         */
        foreach ( $cart->get_cart() as $cart_item_key => $item ) {
            if ( ! empty( $item['gt_bogo_discount'] ) ) {
                $new_price = max( 0, $item['data']->get_price() - $item['gt_bogo_discount'] );
                $item['data']->set_price( $new_price );
            }
        }
    }
}

