<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * GT BOGO Engine - License manager (PRO).
 *
 * Design goals:
 * - Fail-safe: never fatal if remote endpoint is unreachable.
 * - Hardened: PRO features require an active license or a recent cached "valid" result.
 * - Compatible: preserves legacy option keys used in older builds.
 */
final class GT_BOGO_Engine_License {

    private static $instance = null;

    const OPTION_KEY         = 'gt_bogo_engine_license'; // legacy
    const OPTION_KEY_ALT     = 'gt_bogo_engine_license_key'; // newer builds
    const OPTION_STATUS      = 'gt_bogo_engine_license_status';
    const TRANSIENT_STATUS   = 'gt_bogo_engine_license_status_t';

    // Allow brief offline grace to prevent accidental lockouts from temporary DNS issues.
    const GRACE_SECONDS      = 7 * DAY_IN_SECONDS;

    // How often to re-check remotely (if possible).
    const REMOTE_CHECK_TTL   = 12 * HOUR_IN_SECONDS;

    private function __construct() {}

    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Returns the license key (trimmed). Empty string if none.
     */
    public function get_key() {
        $key = (string) get_option( self::OPTION_KEY, '' );
        if ( '' === trim( $key ) ) {
            $key = (string) get_option( self::OPTION_KEY_ALT, '' );
        }
        $key = trim( $key );
        return $key;
    }

    /**
     * Persist the license key.
     */
    public function set_key( $key ) {
        $key = is_string( $key ) ? trim( $key ) : '';
        update_option( self::OPTION_KEY, $key, false );
        update_option( self::OPTION_KEY_ALT, $key, false );
        // Any key change invalidates cached status so we re-check.
        delete_transient( self::TRANSIENT_STATUS );
        delete_option( self::OPTION_STATUS );
    }

    /**
     * Returns the cached status array.
     */
    public function get_status() {
        $cached = get_transient( self::TRANSIENT_STATUS );
        if ( is_array( $cached ) ) {
            return $cached;
        }
        $status = get_option( self::OPTION_STATUS, array() );
        if ( is_array( $status ) && ! empty( $status ) ) {
            set_transient( self::TRANSIENT_STATUS, $status, self::REMOTE_CHECK_TTL );
            return $status;
        }
        return array();
    }

    /**
     * True if PRO is currently allowed.
     */
    public function is_pro_active() {
        $key = $this->get_key();
        if ( '' === $key ) {
            return false;
        }

        $status = $this->get_status();
        if ( empty( $status ) || empty( $status['valid'] ) ) {
            return false;
        }

        // If stored key was removed/tampered with but status remains.
        if ( empty( $status['key_hash'] ) || ! hash_equals( $status['key_hash'], $this->hash_key_for_site( $key ) ) ) {
            return false;
        }

        if ( ! empty( $status['expires'] ) && is_numeric( $status['expires'] ) ) {
            $expires = (int) $status['expires'];
            if ( $expires > 0 && time() > $expires ) {
                return false;
            }
        }

        // If remote check is very stale, allow only within grace window.
        $last = isset( $status['last_checked'] ) ? (int) $status['last_checked'] : 0;
        if ( $last > 0 && ( time() - $last ) > self::GRACE_SECONDS ) {
            // Past grace: require a fresh re-check to keep PRO active.
            return false;
        }

        return true;
    }

    /**
     * Validate license, using remote endpoint when needed. Returns status array.
     * If remote unreachable, uses cached status (within grace) to avoid false negatives.
     */
    public function ensure_validated() {
        $key = $this->get_key();
        if ( '' === $key ) {
            $status = array(
                'valid'        => false,
                'message'      => 'No license key set.',
                'last_checked' => time(),
                'expires'      => 0,
                'key_hash'     => '',
            );
            $this->persist_status( $status );
            return $status;
        }

        $status = $this->get_status();
        $last   = isset( $status['last_checked'] ) ? (int) $status['last_checked'] : 0;

        // Don't hammer the endpoint.
        if ( $last > 0 && ( time() - $last ) < self::REMOTE_CHECK_TTL && ! empty( $status ) ) {
            return $status;
        }

        $remote = $this->remote_check( $key );

        if ( is_wp_error( $remote ) ) {
            // Remote unreachable: keep last known good (within grace).
            if ( ! empty( $status ) && ! empty( $status['valid'] ) && isset( $status['last_checked'] ) ) {
                if ( ( time() - (int) $status['last_checked'] ) <= self::GRACE_SECONDS ) {
                    // Refresh last_checked to extend short TTL only, not grace.
                    $status['message'] = 'Temporarily unable to reach license server. Using cached license status.';
                    $this->persist_status( $status, false );
                    return $status;
                }
            }

            $fallback = array(
                'valid'        => false,
                'message'      => 'Unable to verify license (network error).',
                'last_checked' => time(),
                'expires'      => 0,
                'key_hash'     => $this->hash_key_for_site( $key ),
            );
            $this->persist_status( $fallback );
            return $fallback;
        }

        $this->persist_status( $remote );
        return $remote;
    }

    /**
     * Perform remote validation. Expected JSON response:
     * { "valid": true/false, "expires": 1735689600, "message": "...", "plan": "pro" }
     */
    private function remote_check( $key ) {
        $url = apply_filters( 'gt_bogo_engine_license_api_url', 'https://graphictshirts.shop/bogo/license-check.php' );

        $body = array(
            'license_key'   => $key,
            'site_url'      => home_url(),
            'plugin'        => 'gt-bogo-engine',
            'plugin_ver'    => defined( 'GT_BOGO_ENGINE_VERSION' ) ? GT_BOGO_ENGINE_VERSION : '',
            'wp_ver'        => get_bloginfo( 'version' ),
            'php_ver'       => PHP_VERSION,
            'wc_ver'        => defined( 'WC_VERSION' ) ? WC_VERSION : '',
        );

        $args = array(
            'timeout'   => 15,
            'headers'   => array(
                'Accept' => 'application/json',
            ),
            'body'      => $body,
            'sslverify' => true,
        );

        $resp = wp_remote_post( $url, $args );
        if ( is_wp_error( $resp ) ) {
            return $resp;
        }

        $code = (int) wp_remote_retrieve_response_code( $resp );
        $raw  = (string) wp_remote_retrieve_body( $resp );

        if ( $code < 200 || $code >= 300 || '' === $raw ) {
            return new WP_Error( 'gt_bogo_license_http', 'Bad response from license server.' );
        }

        $json = json_decode( $raw, true );
        if ( ! is_array( $json ) ) {
            return new WP_Error( 'gt_bogo_license_json', 'Invalid JSON from license server.' );
        }

        $valid   = ! empty( $json['valid'] );
        $expires = isset( $json['expires'] ) && is_numeric( $json['expires'] ) ? (int) $json['expires'] : 0;
        $msg     = isset( $json['message'] ) ? (string) $json['message'] : ( $valid ? 'License active.' : 'License invalid.' );

        $status = array(
            'valid'        => (bool) $valid,
            'expires'      => $expires,
            'plan'         => isset( $json['plan'] ) ? (string) $json['plan'] : '',
            'message'      => $msg,
            'last_checked' => time(),
            'key_hash'     => $this->hash_key_for_site( $key ),
        );

        return $status;
    }

    private function persist_status( array $status, $update_last_checked = true ) {
        if ( ! isset( $status['last_checked'] ) || $update_last_checked ) {
            $status['last_checked'] = time();
        }
        update_option( self::OPTION_STATUS, $status, false );
        set_transient( self::TRANSIENT_STATUS, $status, self::REMOTE_CHECK_TTL );
    }

    private function hash_key_for_site( $key ) {
        $salt = (string) wp_salt( 'auth' );
        return hash( 'sha256', strtolower( trim( $key ) ) . '|' . home_url() . '|' . $salt );
    }
}
