<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Front-end rules engine for GT BOGO.
 */
class GT_BOGO_Rules {

    private static $instance = null;
    private $option_key = 'gt_bogo_engine_rules';
    private $settings_key = 'gt_bogo_engine_settings';

    public static function instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Apply discounts as negative fees in cart.
        add_action( 'woocommerce_cart_calculate_fees', array( $this, 'apply_bogo_discounts' ), 20, 1 );
    }

    /**
     * Apply all enabled BOGO rules to the cart.
     */
    public function apply_bogo_discounts( $cart ) {
        if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
            return;
        }

        if ( ! $cart || $cart->is_empty() ) {
            return;
        }

        $rules    = get_option( $this->option_key, array() );
        $settings = get_option( $this->settings_key, array() );
        $debug    = ! empty( $settings['enable_debug'] );

        if ( empty( $rules ) ) {
            return;
        }

        $cart_subtotal = (float) $cart->get_subtotal();

        foreach ( $rules as $rule ) {
            if ( empty( $rule['enabled'] ) ) {
                continue;
            }

            $type         = isset( $rule['type'] ) ? $rule['type'] : 'buy_x_get_y_percent';
            $buy_qty      = isset( $rule['buy_qty'] ) ? max( 1, (int) $rule['buy_qty'] ) : 1;
            $get_qty      = isset( $rule['get_qty'] ) ? max( 1, (int) $rule['get_qty'] ) : 1;
            $discount_val = isset( $rule['discount'] ) ? (float) $rule['discount'] : 0;
            $scope        = isset( $rule['scope'] ) ? $rule['scope'] : 'all';
            $categories   = isset( $rule['categories'] ) && is_array( $rule['categories'] ) ? $rule['categories'] : array();
            $products_raw = isset( $rule['products'] ) ? $rule['products'] : '';
            $min_subtotal = isset( $rule['min_subtotal'] ) ? (float) $rule['min_subtotal'] : 0;
            $start_date   = isset( $rule['start_date'] ) ? $rule['start_date'] : '';
            $end_date     = isset( $rule['end_date'] ) ? $rule['end_date'] : '';

            // Date range check
            if ( $start_date ) {
                $start_ts = strtotime( $start_date . ' 00:00:00' );
                if ( $start_ts && time() < $start_ts ) {
                    continue;
                }
            }
            if ( $end_date ) {
                $end_ts = strtotime( $end_date . ' 23:59:59' );
                if ( $end_ts && time() > $end_ts ) {
                    continue;
                }
            }

            // Min subtotal condition
            if ( $min_subtotal > 0 && $cart_subtotal < $min_subtotal ) {
                continue;
            }

            // Get matching items from cart
            $matching_items = $this->get_matching_cart_items( $cart, $scope, $categories, $products_raw );

            if ( empty( $matching_items ) ) {
                continue;
            }

            $discount_total = 0;

            switch ( $type ) {
                case 'buy_x_get_y_percent':
                    $discount_total = $this->calculate_buy_x_get_y_percent( $matching_items, $buy_qty, $get_qty, $discount_val );
                    break;

                case 'buy_x_get_y_free':
                    // Treat as 100% for the "get" portion.
                    $discount_total = $this->calculate_buy_x_get_y_percent( $matching_items, $buy_qty, $get_qty, 100 );
                    break;

                case 'cheapest_percent':
                    $discount_total = $this->calculate_cheapest_percent( $matching_items, $buy_qty + $get_qty, $discount_val );
                    break;
            }

            if ( $discount_total > 0 ) {
                $label = isset( $rule['name'] ) && $rule['name'] ? $rule['name'] : __( 'BOGO discount', 'gt-bogo-engine' );
                $cart->add_fee( $label, -1 * $discount_total );

                if ( $debug ) {
                    error_log( sprintf( '[GT BOGO] Rule "%s" applied discount: %s', $label, $discount_total ) );
                }
            }
        }
    }

    /**
     * Gather cart items that match scope.
     */
    private function get_matching_cart_items( $cart, $scope, $categories, $products_raw ) {
        $items = array();

        // Parse product IDs from comma string if needed.
        $product_ids = array();
        if ( $products_raw ) {
            $parts = explode( ',', $products_raw );
            foreach ( $parts as $part ) {
                $id = (int) trim( $part );
                if ( $id > 0 ) {
                    $product_ids[] = $id;
                }
            }
        }

        foreach ( $cart->get_cart() as $key => $item ) {
            if ( empty( $item['data'] ) || ! is_a( $item['data'], 'WC_Product' ) ) {
                continue;
            }

            $product   = $item['data'];
            $product_id = $product->get_id();
            $qty       = $item['quantity'];
            $line_price = (float) $product->get_price();

            // Skip free or weirdly priced items
            if ( $qty <= 0 || $line_price <= 0 ) {
                continue;
            }

            $matches = false;

            if ( $scope === 'all' ) {
                $matches = true;
            } elseif ( $scope === 'categories' && ! empty( $categories ) ) {
                $product_cats = wc_get_product_cat_ids( $product_id );
                if ( array_intersect( $categories, $product_cats ) ) {
                    $matches = true;
                }
            } elseif ( $scope === 'products' && ! empty( $product_ids ) ) {
                if ( in_array( $product_id, $product_ids, true ) ) {
                    $matches = true;
                }
            }

            if ( $matches ) {
                $items[] = array(
                    'key'   => $key,
                    'id'    => $product_id,
                    'qty'   => $qty,
                    'price' => $line_price,
                );
            }
        }

        return $items;
    }

    /**
     * Buy X Get Y at % off.
     *
     * - Count total units among matching items.
     * - For each group of (X+Y), discount Y items at %.
     */
    private function calculate_buy_x_get_y_percent( $items, $buy_qty, $get_qty, $discount_percent ) {
        if ( $discount_percent <= 0 ) {
            return 0;
        }

        $total_units = 0;
        foreach ( $items as $item ) {
            $total_units += $item['qty'];
        }

        $group_size = $buy_qty + $get_qty;
        if ( $group_size <= 0 ) {
            return 0;
        }

        $eligible_groups = floor( $total_units / $group_size );
        if ( $eligible_groups <= 0 ) {
            return 0;
        }

        $total_get_units = $eligible_groups * $get_qty;

        // Sort items by price DESC so we discount the cheapest units last (more conservative).
        usort( $items, function ( $a, $b ) {
            if ( $a['price'] === $b['price'] ) {
                return 0;
            }
            return ( $a['price'] < $b['price'] ) ? 1 : -1;
        } );

        $remaining = $total_get_units;
        $discount_total = 0;

        // We'll discount from the cheapest side: that is, reverse the array.
        $items_reversed = array_reverse( $items );

        foreach ( $items_reversed as $item ) {
            if ( $remaining <= 0 ) {
                break;
            }

            $unit_price = $item['price'];
            $qty_available = $item['qty'];

            $qty_to_discount = min( $remaining, $qty_available );
            $discount_total += ( $unit_price * $qty_to_discount ) * ( $discount_percent / 100 );
            $remaining -= $qty_to_discount;
        }

        return $discount_total;
    }

    /**
     * Cheapest item % off when threshold reached.
     *
     * - If total units >= threshold.
     * - Find the cheapest unit and apply % discount to one unit per threshold reached.
     */
    private function calculate_cheapest_percent( $items, $threshold_qty, $discount_percent ) {
        if ( $discount_percent <= 0 ) {
            return 0;
        }

        $total_units = 0;
        $all_units   = array();

        foreach ( $items as $item ) {
            $total_units += $item['qty'];
            for ( $i = 0; $i < $item['qty']; $i++ ) {
                $all_units[] = $item['price'];
            }
        }

        if ( $total_units < $threshold_qty ) {
            return 0;
        }

        sort( $all_units, SORT_NUMERIC ); // ascending
        $groups = floor( $total_units / $threshold_qty );
        $units_to_discount = $groups; // discount one cheapest item per group

        $discount_total = 0;
        for ( $i = 0; $i < $units_to_discount && $i < count( $all_units ); $i++ ) {
            $discount_total += $all_units[ $i ] * ( $discount_percent / 100 );
        }

        return $discount_total;
    }
}

