<?php
/**
 * Core Engine – GT BOGO ENGINE for WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class GT_BOGO_Engine {

    private static $instance = null;

    /**
     * Singleton instance
     */
    public static function instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {

        // Load admin
        if ( is_admin() ) {
            require_once GT_BOGO_ENGINE_DIR . 'includes/class-gt-bogo-engine-admin.php';
            new GT_BOGO_Engine_Admin();
        }

        // Apply discounts
        add_action( 'woocommerce_before_calculate_totals', array( $this, 'apply_discounts' ), 9999 );
    }


    /**
     * MAIN ENGINE – APPLY BOGO DISCOUNTS
     */
    public function apply_discounts( $cart ) {

        if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
            return;
        }

        if ( ! $cart instanceof WC_Cart ) {
            return;
        }

        $settings = get_option( 'gt_bogo_engine_settings', array() );
        $free_rule = get_option( 'gt_bogo_engine_free_rule', array() );
        $pro_rules = get_option( 'gt_bogo_engine_rules', array() );

        // engine disabled?
        if ( empty( $settings['enable_engine'] ) ) {
            return;
        }

        /**
         * Build cart quantity map
         */
        $cart_items = array();

        foreach ( $cart->get_cart() as $key => $item ) {
            if ( ! isset( $item['product_id'] ) ) continue;

            $product_id = $item['product_id'];
            $qty        = $item['quantity'];
            $price      = $item['data']->get_price();

            $cart_items[] = array(
                'key'        => $key,
                'product_id' => $product_id,
                'qty'        => $qty,
                'price'      => $price,
            );
        }

        /**
         * PROCESS FREE RULE (Global)
         */
        if ( isset( $free_rule['enabled'] ) && $free_rule['enabled'] ) {
            $this->apply_single_rule( $cart, $cart_items, $free_rule, true );
        }

        /**
         * PROCESS PRO RULES
         */
        if ( is_array( $pro_rules ) && ! empty( $pro_rules ) ) {
            foreach ( $pro_rules as $rule ) {
                if ( ! isset( $rule['enabled'] ) || ! $rule['enabled'] ) continue;

                $this->apply_single_rule( $cart, $cart_items, $rule, false );
            }
        }
    }


    /**
     * Apply a single BOGO rule
     */
    private function apply_single_rule( $cart, $cart_items, $rule, $is_free_rule = false ) {

        $buy_qty   = max( 1, intval( $rule['buy_qty'] ?? 1 ) );
        $get_qty   = max( 1, intval( $rule['get_qty'] ?? 1 ) );
        $disc_type = $rule['discount_type'] ?? 'percent';
        $disc_val  = floatval( $rule['discount_value'] ?? 0 );
        $scope     = $rule['scope'] ?? 'global';

        // determine eligible products
        $eligible_products = array();

        foreach ( $cart_items as $item ) {

            // category scope
            if ( $scope === 'categories' && ! empty( $rule['categories'] ) ) {
                $cats = wc_get_product_term_ids( $item['product_id'], 'product_cat' );
                if ( empty( array_intersect( $cats, $rule['categories'] ) ) ) continue;
            }

            // product scope
            if ( $scope === 'products' && ! empty( $rule['products'] ) ) {
                if ( ! in_array( $item['product_id'], $rule['products'], true ) ) continue;
            }

            // add to eligible list
            for ( $i = 0; $i < $item['qty']; $i++ ) {
                $eligible_products[] = array(
                    'key'   => $item['key'],
                    'price' => $item['price'],
                );
            }
        }

        // Not enough items? stop rule
        if ( count( $eligible_products ) < ( $buy_qty + $get_qty ) ) {
            return;
        }

        /**
         * Sort lowest priced items first (standard BOGO behavior)
         */
        usort( $eligible_products, function( $a, $b ) {
            return $a['price'] <=> $b['price'];
        });

        /**
         * For every group, discount the GET items
         */
        $group_size = $buy_qty + $get_qty;
        $groups = floor( count( $eligible_products ) / $group_size );

        for ( $g = 0; $g < $groups; $g++ ) {

            // Discount the "get" items at end of each group
            $offset = ( $g * $group_size ) + $buy_qty;

            for ( $i = 0; $i < $get_qty; $i++ ) {

                $item_index = $offset + $i;
                if ( ! isset( $eligible_products[ $item_index ] ) ) continue;

                $key   = $eligible_products[ $item_index ]['key'];
                $price = $eligible_products[ $item_index ]['price'];

                $discount = 0;

                if ( $disc_type === 'percent' ) {
                    $discount = ( $price * $disc_val / 100 );
                } elseif ( $disc_type === 'fixed' ) {
                    $discount = min( $disc_val, $price );
                }

                if ( $discount > 0 ) {
                    $cart->cart_contents[ $key ]['gt_bogo_discount'] =
                        ( $cart->cart_contents[ $key ]['gt_bogo_discount'] ?? 0 ) + $discount;
                }
            }
        }

        /**
         * Apply discount to cart items
         */
        foreach ( $cart->get_cart() as $key => $item ) {
            if ( isset( $item['gt_bogo_discount'] ) ) {
                $item['data']->set_price( max( 0, $item['data']->get_price() - $item['gt_bogo_discount'] ) );
            }
        }
    }
}
